//
//  UIComponentTests.swift
//  death_app Watch AppTests
//
//  Created by Task Master on 2025-09-16.
//

import XCTest
import SwiftUI
@testable import death_app_Watch_App

final class UIComponentTests: XCTestCase {
    
    // MARK: - Countdown Timer Display Tests
    
    func testCountdownTimerFormatting() throws {
        let countdownFormatter = CountdownFormatter()
        
        // Test years
        let years = countdownFormatter.formatTimeRemaining(totalSeconds: 31536000) // 1 year
        XCTAssertTrue(years.contains("1"), "Should show 1 year")
        XCTAssertTrue(years.contains("year"), "Should include year unit")
        
        // Test days
        let days = countdownFormatter.formatTimeRemaining(totalSeconds: 86400) // 1 day
        XCTAssertTrue(days.contains("1"), "Should show 1 day")
        XCTAssertTrue(days.contains("day"), "Should include day unit")
        
        // Test hours
        let hours = countdownFormatter.formatTimeRemaining(totalSeconds: 3600) // 1 hour
        XCTAssertTrue(hours.contains("1"), "Should show 1 hour")
        XCTAssertTrue(hours.contains("hour"), "Should include hour unit")
        
        // Test minutes
        let minutes = countdownFormatter.formatTimeRemaining(totalSeconds: 60) // 1 minute
        XCTAssertTrue(minutes.contains("1"), "Should show 1 minute")
        XCTAssertTrue(minutes.contains("minute"), "Should include minute unit")
        
        // Test seconds
        let seconds = countdownFormatter.formatTimeRemaining(totalSeconds: 30) // 30 seconds
        XCTAssertTrue(seconds.contains("30"), "Should show 30 seconds")
        XCTAssertTrue(seconds.contains("second"), "Should include second unit")
    }
    
    func testCountdownTimerEdgeCases() throws {
        let countdownFormatter = CountdownFormatter()
        
        // Test zero
        let zero = countdownFormatter.formatTimeRemaining(totalSeconds: 0)
        XCTAssertTrue(zero.contains("0"), "Should handle zero")
        
        // Test negative (expired)
        let negative = countdownFormatter.formatTimeRemaining(totalSeconds: -100)
        XCTAssertTrue(negative.contains("Expired") || negative.contains("0"), "Should handle negative values")
        
        // Test very large numbers
        let large = countdownFormatter.formatTimeRemaining(totalSeconds: 3153600000) // ~100 years
        XCTAssertNotNil(large, "Should handle large values")
        XCTAssertFalse(large.isEmpty, "Should not return empty string")
    }
    
    func testCountdownTimerPrecision() throws {
        let countdownFormatter = CountdownFormatter()
        
        // Test compound time (1 year, 30 days, 5 hours)
        let compound = countdownFormatter.formatTimeRemaining(totalSeconds: 31536000 + 2592000 + 18000)
        XCTAssertTrue(compound.contains("1") && compound.contains("year"), "Should include years")
        
        // Test precision setting
        countdownFormatter.precision = .seconds
        let precise = countdownFormatter.formatTimeRemaining(totalSeconds: 3661) // 1h 1m 1s
        let components = precise.components(separatedBy: " ")
        XCTAssertGreaterThan(components.count, 2, "Should show multiple components with high precision")
    }
    
    // MARK: - Visual Indicator Tests
    
    func testHealthStatusIndicator() throws {
        let indicator = HealthStatusIndicator()
        
        // Test healthy status
        let healthyColor = indicator.colorForHealthScore(0.9)
        XCTAssertEqual(healthyColor, .green, "High health score should be green")
        
        // Test moderate status
        let moderateColor = indicator.colorForHealthScore(0.6)
        XCTAssertEqual(moderateColor, .yellow, "Moderate health score should be yellow")
        
        // Test poor status
        let poorColor = indicator.colorForHealthScore(0.3)
        XCTAssertEqual(poorColor, .red, "Low health score should be red")
        
        // Test edge cases
        let maxColor = indicator.colorForHealthScore(1.0)
        XCTAssertEqual(maxColor, .green, "Maximum score should be green")
        
        let minColor = indicator.colorForHealthScore(0.0)
        XCTAssertEqual(minColor, .red, "Minimum score should be red")
    }
    
    func testRiskFactorVisualIndicators() throws {
        let riskIndicator = RiskFactorIndicator()
        
        // Test smoking risk
        let smokingIcon = riskIndicator.iconForRiskFactor(.smoking, value: 1.0)
        XCTAssertEqual(smokingIcon, "lungs.fill", "Smoking should use lungs icon")
        
        let smokingColor = riskIndicator.colorForRiskFactor(.smoking, value: 1.0)
        XCTAssertEqual(smokingColor, .red, "Active smoking should be red")
        
        // Test exercise risk
        let exerciseIcon = riskIndicator.iconForRiskFactor(.exercise, value: 0.0)
        XCTAssertEqual(exerciseIcon, "figure.walk", "Exercise should use walk icon")
        
        let sedentaryColor = riskIndicator.colorForRiskFactor(.exercise, value: 0.0)
        XCTAssertEqual(sedentaryColor, .red, "No exercise should be red")
        
        let activeColor = riskIndicator.colorForRiskFactor(.exercise, value: 1.0)
        XCTAssertEqual(activeColor, .green, "High exercise should be green")
        
        // Test BMI risk
        let bmiIcon = riskIndicator.iconForRiskFactor(.bmi, value: 25.0)
        XCTAssertEqual(bmiIcon, "scalemass", "BMI should use scale icon")
        
        let normalBMIColor = riskIndicator.colorForRiskFactor(.bmi, value: 22.0)
        XCTAssertEqual(normalBMIColor, .green, "Normal BMI should be green")
        
        let obeseBMIColor = riskIndicator.colorForRiskFactor(.bmi, value: 35.0)
        XCTAssertEqual(obeseBMIColor, .red, "Obese BMI should be red")
    }
    
    func testPredictionConfidenceIndicator() throws {
        let confidenceIndicator = PredictionConfidenceIndicator()
        
        // Test high confidence
        let highConfidenceIcon = confidenceIndicator.iconForConfidence(0.9)
        XCTAssertEqual(highConfidenceIcon, "checkmark.circle.fill", "High confidence should show checkmark")
        
        let highConfidenceColor = confidenceIndicator.colorForConfidence(0.9)
        XCTAssertEqual(highConfidenceColor, .green, "High confidence should be green")
        
        // Test medium confidence
        let mediumConfidenceIcon = confidenceIndicator.iconForConfidence(0.6)
        XCTAssertEqual(mediumConfidenceIcon, "exclamationmark.triangle.fill", "Medium confidence should show warning")
        
        let mediumConfidenceColor = confidenceIndicator.colorForConfidence(0.6)
        XCTAssertEqual(mediumConfidenceColor, .yellow, "Medium confidence should be yellow")
        
        // Test low confidence
        let lowConfidenceIcon = confidenceIndicator.iconForConfidence(0.3)
        XCTAssertEqual(lowConfidenceIcon, "questionmark.circle.fill", "Low confidence should show question")
        
        let lowConfidenceColor = confidenceIndicator.colorForConfidence(0.3)
        XCTAssertEqual(lowConfidenceColor, .red, "Low confidence should be red")
    }
    
    // MARK: - UI Responsiveness Tests
    
    func testCountdownViewUpdates() throws {
        let expectation = XCTestExpectation(description: "Countdown view updates")
        
        class MockCountdownView: CountdownView {
            var updateCallCount = 0
            
            override func updateDisplay() {
                super.updateDisplay()
                updateCallCount += 1
            }
        }
        
        let countdownView = MockCountdownView()
        countdownView.startTimer()
        
        DispatchQueue.main.asyncAfter(deadline: .now() + 2.1) {
            countdownView.stopTimer()
            XCTAssertGreaterThan(countdownView.updateCallCount, 1, "Countdown should update multiple times")
            expectation.fulfill()
        }
        
        wait(for: [expectation], timeout: 3.0)
    }
    
    func testButtonStateManagement() throws {
        let buttonManager = ButtonStateManager()
        
        // Test initial state
        XCTAssertFalse(buttonManager.isLoading, "Should start not loading")
        XCTAssertTrue(buttonManager.isEnabled, "Should start enabled")
        
        // Test loading state
        buttonManager.setLoading(true)
        XCTAssertTrue(buttonManager.isLoading, "Should be loading")
        XCTAssertFalse(buttonManager.isEnabled, "Should be disabled when loading")
        
        // Test error state
        buttonManager.setError("Network error")
        XCTAssertFalse(buttonManager.isLoading, "Should not be loading after error")
        XCTAssertTrue(buttonManager.hasError, "Should have error")
        XCTAssertEqual(buttonManager.errorMessage, "Network error", "Should store error message")
        
        // Test reset
        buttonManager.reset()
        XCTAssertFalse(buttonManager.isLoading, "Should not be loading after reset")
        XCTAssertFalse(buttonManager.hasError, "Should not have error after reset")
        XCTAssertTrue(buttonManager.isEnabled, "Should be enabled after reset")
    }
    
    // MARK: - Accessibility Tests
    
    func testVoiceOverSupport() throws {
        let countdownView = CountdownView()
        
        // Test accessibility labels
        XCTAssertNotNil(countdownView.accessibilityLabel, "Should have accessibility label")
        XCTAssertFalse(countdownView.accessibilityLabel?.isEmpty ?? true, "Accessibility label should not be empty")
        
        // Test accessibility hints
        XCTAssertNotNil(countdownView.accessibilityHint, "Should have accessibility hint")
        
        // Test accessibility traits
        XCTAssertTrue(countdownView.accessibilityTraits.contains(.updatesFrequently), "Should mark as frequently updating")
    }
    
    func testDynamicTypeSupport() throws {
        let textFormatter = TextFormatter()
        
        // Test different content size categories
        let smallText = textFormatter.formatTextForContentSize(.small)
        let largeText = textFormatter.formatTextForContentSize(.extraExtraExtraLarge)
        
        XCTAssertNotEqual(smallText.font.pointSize, largeText.font.pointSize, "Should adjust font size for content size")
        XCTAssertLessThan(smallText.font.pointSize, largeText.font.pointSize, "Large content size should have larger font")
    }
    
    func testHighContrastSupport() throws {
        let colorProvider = AccessibilityColorProvider()
        
        // Test normal contrast
        colorProvider.useHighContrast = false
        let normalColor = colorProvider.primaryTextColor()
        
        // Test high contrast
        colorProvider.useHighContrast = true
        let highContrastColor = colorProvider.primaryTextColor()
        
        XCTAssertNotEqual(normalColor, highContrastColor, "Should provide different colors for high contrast")
    }
    
    // MARK: - User Interface Layout Tests
    
    func testWatchScreenSizeAdaptation() throws {
        let layoutManager = WatchLayoutManager()
        
        // Test 40mm watch
        let layout40mm = layoutManager.layoutForScreenSize(CGSize(width: 162, height: 197))
        XCTAssertLessThan(layout40mm.fontSize, 16, "Should use smaller font for 40mm watch")
        XCTAssertLessThan(layout40mm.padding, 20, "Should use smaller padding for 40mm watch")
        
        // Test 44mm watch
        let layout44mm = layoutManager.layoutForScreenSize(CGSize(width: 184, height: 224))
        XCTAssertGreaterThan(layout44mm.fontSize, layout40mm.fontSize, "Should use larger font for 44mm watch")
        XCTAssertGreaterThan(layout44mm.padding, layout40mm.padding, "Should use larger padding for 44mm watch")
    }
    
    func testScrollingBehavior() throws {
        let scrollManager = ScrollBehaviorManager()
        
        // Test scroll position tracking
        scrollManager.updateScrollPosition(0.5)
        XCTAssertEqual(scrollManager.currentScrollPosition, 0.5, "Should track scroll position")
        
        // Test scroll limits
        scrollManager.updateScrollPosition(-0.1)
        XCTAssertGreaterThanOrEqual(scrollManager.currentScrollPosition, 0.0, "Should not scroll below 0")
        
        scrollManager.updateScrollPosition(1.1)
        XCTAssertLessThanOrEqual(scrollManager.currentScrollPosition, 1.0, "Should not scroll above 1")
    }
    
    // MARK: - Animation Tests
    
    func testCountdownAnimations() throws {
        let expectation = XCTestExpectation(description: "Animation completion")
        
        let animator = CountdownAnimator()
        
        animator.animateNumberChange(from: 100, to: 99) {
            XCTAssertTrue(true, "Animation should complete")
            expectation.fulfill()
        }
        
        wait(for: [expectation], timeout: 2.0)
    }
    
    func testPredictionLoadingAnimation() throws {
        let expectation = XCTestExpectation(description: "Loading animation")
        
        let loadingView = PredictionLoadingView()
        loadingView.startAnimating()
        
        DispatchQueue.main.asyncAfter(deadline: .now() + 1.0) {
            XCTAssertTrue(loadingView.isAnimating, "Should be animating")
            loadingView.stopAnimating()
            XCTAssertFalse(loadingView.isAnimating, "Should stop animating")
            expectation.fulfill()
        }
        
        wait(for: [expectation], timeout: 2.0)
    }
    
    // MARK: - Performance Tests
    
    func testUIRenderingPerformance() throws {
        measure {
            let countdownView = CountdownView()
            countdownView.updateTimeRemaining(3600) // 1 hour
            countdownView.layoutIfNeeded()
        }
    }
    
    func testScrollingPerformance() throws {
        let scrollView = PredictionHistoryScrollView()
        
        // Add many items to test scrolling performance
        var predictions: [LifePrediction] = []
        for i in 0..<100 {
            let prediction = LifePrediction(
                lifeExpectancy: 75.0 + Double(i % 20),
                generatedAt: Date().addingTimeInterval(-Double(i * 86400)),
                riskFactors: [:],
                healthMetrics: [:],
                confidenceScore: 0.8
            )
            predictions.append(prediction)
        }
        
        measure {
            scrollView.updatePredictions(predictions)
            scrollView.layoutIfNeeded()
        }
    }
}